State Space Model
=========================================

An object-oriented framework in MATLAB for defining and running linear and non-linear dynamic models in state space form.

## Maintainers
* David Katzin, `david.katzin@wur.nl`

## In this repository

- `readme.md, Workspace.png`  - Readme file and supporting images
- *Code*
   - `example.m` - Basic example for using this framework, see Sections 3.2, 4 below
   - `exampleTimeDep.m` - Example for using time-dependent variables, see Section 5 below
   - *@DynamicElement* - Files for the DynamicElement class
   - *@StateSpaceModel* - Files for the StateSpaceModel class
      - `StateSpaceModel.m` - Defines the StateSpaceModel class
	  - `copySsm.m` - Create a copy of a StateSpaceModel object
	  - `addParam.m` - For defining parameters when setting up the model
	  - `runSim.m, rungeKutta.m, rungeKuttaNoAux.m, solveTomlab.m` - Functions for running the model
	  - `getFieldNames.m` - Used internally by class functions
	  - `plot.m, trapz.m` - Functions for viewing and analyzing the model after simulation
	  - `changeRes.m, concat.m, cutTime.m` - Functions for restructuring the model after simulation
   - *Other functions* - Useful functions that are not part of the class definitions
      - *CmdLineProgressBar* - Visualize running time during simulation. By Ita Katz (with some modifications)
      - `ifElse.m` - Create a DynamicElement based on a logical expression
      - `proportionalControl.m` - Create a DynamicElement which responds proportionally to disturbances

## 1. Background

### 1.1 Introduction

This platform allows to design and simulate models that are defined as [state space models](https://en.wikipedia.org/wiki/State-space_representation), i.e., models defined by ordinary differential equations (ODEs), both linear and non-linear. Three principals lie at the core of this platform:
1. **Clarity:** The models designed in this framework can be easily read, in particular, the underlying functions, i.e., the ODEs defining the model, are explicit.
2. **Modularity:** In this framework it is easy to replace one model for another, modify a single function within a model, or modify the way models are solved, without rewriting big pieces of code.
3. **Separability:** There is a clear distinction between a description of a model (its ODEs, parameters, interpretation) and the model solvers.

The above principles make this platform stand out from other modeling platforms in the sense that models defined in this platform are easy to create, understand, and modify. An object representing a model contains within it all the definitions, equations, parameters and values for the model. These can (and should!) be accompanied by documentation giving references (e.g., literature) to each decision made during model design. After simulating, all the simulated data and outputs are automatically stored in a structured manner. All of the above is kept separate from the methods or scripts dedicated to solving/simulating the model, which are kept at an abstract level that can be applied to any model. In other words, you can focus only on your model without worrying about how to represent the model in code, how to save important information about each simulation, or about the technicalities of solving and simulating. Alternatively, anyone can develop new methods for solving or analyzing models which are immediately applicable to all models developed within this framework.

Several definitions are needed in the upcoming sections:

- **Model definition** is the set of ordinary differential equations describing the model. The number of equations is equal to the number of states in the model.
- **Model solving** refers to solving the boundary-value problem, i.e., the model ODEs together with initial values for all states. In other words, this is **simulating** the model. In the case of optimal control (OC) problems, this also includes solving the OC problem, i.e., finding an optimal solution.
- **Smoothness and stiffness** are properties of the model definitions. These are relative terms. Mathematically, a function that is more smooth has more continuous derivatives. Stiffness is the opposite of smoothness, i.e., a stiff function has few (or no) continuous derivatives. Some stiff functions that regularly occur in modeling are absolute value (near zero), square root (near zero), and if/else statements.

### 1.2 State Space Models

In this section, we present a general description of [state space models](https://en.wikipedia.org/wiki/State-space_representation). These models are composed of the following:

- **Time $(t)$** - The variable that is being derived in the ODEs. Typically the model starts at $t=0$ and ends at some predefined end point $t=t_f$. However there are state space models with variable end points, i.e., $t_f$ depends on the states.
- **Parameters $(p)$** (*constants*) are any variables that do not change throughout the model run. It is generally better to avoid using numbers in your model definition. Give any consant number a variable name and assign it as a parameter.
- **States $(x)$** (also known as *state variables*) are the variables in the model that are defined by ODEs that depend on the time variable $t$. Thus an ODE for the state $x$ would look something like: 
$$\frac{dx}{dt} = f(x,t)$$
For every state in the model there is one ODE, and vice versa.
- **Auxilliary states $(a)$** (also known as *semi-states*) are variables that depend on the time variable $t$ and also possible on other model variables (states, inputs, parameters, other auxiliary states), but do not have their own ODE. These are typically variables that need to be tracked for some reason, or that make the ODEs more easy to read. For instance, we might have a state $x$ and auxilliary states $a_1$ and $a_2$, where

$$\frac{dx}{dt} = a_1(t)+a_2(t)$$
$$a_1(t) = \sin (t)$$
$$a_2(t) = \cos (t)$$

- **Inputs $(d)$** (also known as *disturbances* or *uncontrolled inputs*) are variables that depend on the time variable $t$ only. They are external to the model, can influence it, but and are not influenced by it.
- **Controls $(u)$** (also termed *controlled inputs*) represent variables that can be controlled by the system. For example in a greenhouse model, the weather may be considered as an uncontrolled input $(d)$, but the greenhouse actuators (e.g., window opening) is a controlled input $(u)$.
- **Constraints $(c)$** (or *bounds*) A set of equalities and inequalities which the states and controls must maintain. Important for optimization problems.
- **Goal function $(g)$** (or *objective function*) in the case of optimization problems, this function defines the optimality criterion.

*Note 1:* A state space model can be constructed with simply a time variable and a single state variable. All other components of the model listed above are optional. 

*Note 2:* Typically, state space models are composed of linear equations and can be represented by matrices. However, with the above definition that doesn't have to be the case. Non-linear equations also work just fine. 

### 1.3 Object oriented programming

[Object-oriented programming (OOP)](https://en.wikipedia.org/wiki/Object-oriented_programming) is "a programming paradigm based on the concept of "objects", which may contain data, in the form of fields, often known as attributes; and code, in the form of procedures, often known as methods" ([Wikipedia](https://en.wikipedia.org/wiki/Object-oriented_programming)). It is a cornerstone of modern programming, and is especially useful for the principal of separability described above. In the state space modelling framework, a model (its definition) is defined as an object, and a solver is a method which acts on this object.

#### Variable classes
In MATLAB, every declared variable belongs to a *class*. This is not always explicit, but the class determines how the variable is stored in memory and how functions act on it. As a simple example, try the following in MATLAB:

> <code> a = 0;<br>
> b = '0'; <br>
> c = "0"; <br>

In your workspace, you can see the classes of current variables:

![Workspace](Workspace.png) 

You can see that <code>a</code> belongs to the class "double" (a format for storing real numbers), <code>b</code> bleongs to the class "char" (a format for storing characters), and <code>c</code> bleongs to the class "string" (a format for storing text strings). This might not mean a lot, but try the following:

> <code> >> a+5

> <code> ans =

> <code> 5

> <code> >> b+5

> <code> ans =

> <code> 53

> <code> >> c+5

> <code> ans =

> <code> "05"


It's not important to figure out why these seemingly strange answers come out, what matters here is that the function "+5" behaves differently for each of these different variable classes. 

In object oriented programming, it is possible to define new variable classes, and methods that act on these classes. A variable that is made using a custom class is typically called an *instance* or an *object* of that class. Objects of custom classes typically have *attributes*, also called *properties*. These properties are quite similar to MATLAB <code>struct</code>s, with the exception that the object follows a fixed format. 

A new class is created in MATLAB by creating a file with the class name, e.g., <code>ClassName.m</code> (class names typically begin with an uppercase letter), and starting that file with the expression<br>
<code>classdef ClassName</code><br>

After this expression the symbol <code>< </code> may follow, and some more text which has to do with *class inheritance*, which is not covered in this tutorial. All files related to a particular class may be grouped in a folder named <code>@ClassName</code>.

A new instance of a custom class is created using the class *constructor*, which is a function that has the same name as the class. For example, an instance <code>a</code> of the class <code>NewClass</code> will be created as:

>> <code>a = NewClass();

### 1.4 Handle and value classes

There is a distinction in MATLAB between *handle* and *value* classes. Handle classes basically hold an address in memory (this is called a *pointer* in other programming languages), and so passing a handle variable to a function actually passes its location in memory, and any changes to that variable inside a function will influence the variable. This, Handle classes have a sort of global feature, where functions that change the variable do not need to return the argument. Consider two variables, `v` and `h`, each of the respective type of class, and consider the function:

> `function setToZero(arg)`

> > `arg = 0;`

> `end`

Running `setToZero(v)` will not change `v` since `arg` inside the function is a new copy of `v`. However running `setToZero(h)` will change `h`, because `arg` and `h` both point to the same location in memory.

Similarly, setting `h1 = h;` will not create a new variable; any changes made to `h1` will also affect `h`. Assume `h` has a property `val`. The following will happen:

> `h.val = 100;` <br>
> `h1 = h;` <br>
> `h1.val = 0;` <br>
> `h`
> > `HandleClass with properties:`
> > > `val: 0`

This is just an example. It will not work if you try to run it in MATLAB, because the class HandleClass has not been defined. The classes described below, `DynamicElement` and `StateSpaceModel` are handle classes. You can experiment with cases such as above once you create objects of these classes.

Another interesting feature in handle classes is the distinction between the operators `==` and `isequal`. The operator `==` will return `true` only if the objects on each side point to the same place in memory, i.e., changes made to the one will influence the other. The operator `isequal` will return true if both objects have the same values. Thus:

> `h1 = HandleClass();`<br>
> `h2 = HandleClass();`<br>
> `h3 = h1`; <br>
> `h1 == h2`
> > `ans = `<br>
> > `logical`<br>
> > `0`<br>

> `isequal(h1,h2)`<br>
> > `ans = `<br>
> > `logical`<br>
> > `1`<br>

> `h1 == h3`<br>
> > `ans = `<br>
> > `logical`<br>
> > `1`<br>

> `isequal(h1,h3)`<br>
> > `ans = `<br>
> > `logical`<br>
> > `1`


## 2. The DynamicElement class

A DynamicElement represents an entity that changes through time. An object of this class has two properties: 

- <code>def</code> - The definition of the object. This is typically a string containing the name of the object, or an instruction on how it works. For example in the case of states, their <code>def</code> will contain the relevant ODE.
- <code>val</code> - The value(s) of the object's trajectory. These could be either a single number, e.g., in the case of parameters, or a 2-column matrix, where the first column represents time, and the second column represents values. 

The DynamicElement class is a handle class.

### 2.1 Creating a new DynamicElement from scratch
A new DynamicElement is created by using a constructor: <br>
> <code>  de = DynamicElement(def,val);

If no arguments are passed, an empty DynamicElement is created:

> <code>>> de = DynamicElement()<br><br>
> de = <br>
> > <code> DynamicElement with properties:

> > > <code>def: [] <br>
    val: []

If a single argument is given, it will be set as the object's <code>def</code> (a string is expected):

> <code>>> de = DynamicElement('definition')<br><br>
> de = <br>
> > <code> DynamicElement with properties:

> > > <code>def: 'definition' <br>
>    val: []

If two arguments are given, they will be set as the object's <code>def</code> and <code>val</code> (a string and number are expected, or a string and a number array with 2 columns):

> <code>>> de = DynamicElement('definition',0)<br><br>
> de = <br>
> > <code> DynamicElement with properties:

> > > <code>def: 'definition' <br>
>    val: 0

> <code>>> de = DynamicElement('definition',[0 0; 1 100])<br><br>
> de = <br>
> > <code> DynamicElement with properties:

> > > <code>def: 'definition' <br>
>    val: [2×2 double]

### 2.2 Creating a new DynamicElement by copying a DynamicElement

The constructor for the DynamicElement class will also accept a DynamicElement argument. This will create a new DynamicElement with the same values as the passed argument. Try the following:

> `d1 = DynamicElement();`<br>
> `d1.def = 'definition';`<br>
> `d2 = DynamicElement(d1);`<br>
> `d3 = d1;`

Now try some of the things described in Section 1.4. You'll see that `d2` is a copy of `d1`, i.e., identical to it, but not the same object. In contrast, `d1` and `d3` point to the same place in memory; changing one of them will influence the other.

### 2.3 Creating a new DynamicElement with mathematical operations

A new DynamicElement can also be created by using one or two existing DynamicElements and mathematical operations:

> <code>>> a = DynamicElement('a',1);<br>
> <code>>> b = DynamicElement('b',2);<br>
> <code>>> c = a+b

> <code> c = <br>
> > <code> DynamicElement with properties:
> > >  <code>  def: '(a)+(b)'<br>
    val: 3
	
Typically, these arithmetic operations between DynamicElements behave as one might expect: the definition is composed of the two definitions of the objects being used (the operands), and the value is a result of the values of the two objects. However some care should be taken:

- If one of the operands has an empty <code>val</code>, the result will have an empty <code>val</code>.
- If both operands have a scalar (single number) as their <code>val</code>, the result will have a scalar <code>val</code>, with its value a result of the arithmetic operation.
- If both operands have as their <code>val</code> a 2-column matrix *with the same values in their first column*, the new object will also have as its <code>val</code> a 2-column matrix with the first column identical to those of the operands. The second column will be the result of the arithmetic operation between the second columns of the two operands.
- If one operand as a scalar as its <code>val</code> and the other operand has a 2-column matrix as its <code>val</code>, the result will have as its <code>val</code> a 2-column matrix, where the first one is identical to the first column of the operans, and the second is the result of the mathematical operation between the scalar and the second column of the other operand.
- In all other cases, a mathematical operation will not be possible, the results will have an empty <code>val</code>, and a warning will be issued.

The currently available operations for DynamicElements are:
#### Binary operations (between two DynamicElements):
<code>+, -, \*, /, ^, .\*, ./, .\^, max, min, \> , \<, \<=, \>=, ==, ~=, &, |</code>

If you don't know the difference between <code>\*</code> and <code>.\*</code> in MATLAB (matrix multiplication and element-wise multiplication), please look it up! If you're still not sure you understand, chances are you'd be safe always using <code>.\*, ./, .\^ </code> instead of <code>\*, /, \^</code>.

#### Unary operations (on one DynamicElement):
<code>abs, exp, sqrt, floor, ceil, sign, nthroot, sin, cos, sind, cosd, mod, ~

### 2.3 Other functions for DynamicElements

<code> plot(de)</code> - Creates a plot of the trajectory of the DynamicElement. This is equivalent to: <br> 
<code>plot(de.val(:,1),de.val(:,2))</code>.

<code> trapz(de)</code> - Calculates the area under the curve of the trajectory. This is equivalent to: <br> <code>trapz(de.val(:,1),de.val(:,2))</code>.


## 3. The StateSpaceModel class

A StateSpaceModel is a model in state space form. It has the following attributes:

- <code>x</code> - The model states. The property <code>x</code> should be defined as a struct containing DynamicElements representing the model states, e.g.:

>> <code> x.state1 = DynamicElement();<br>
>> x.state2 = DynamicElement();

- <code>a</code> - Auxiliary states. As with the states, this is a struct containing DynamicElements.
- <code>d</code> - Inputs. A struct of DynamicElements.
- <code>p</code> - Parameters. A struct of DynamicElements.
- <code>u</code> - Controls. A struct of DynamicElements.
- <code>c</code> - Constraints. See below.
- <code>g</code> - Goal function. See below.
- <code>t</code> - Time span. For the sake of consistency, also a DynamicElement. In this case, the <code>def</code> should be a string representing when the time span begins (e.g., <code>'01-Jan-2016 23:55'</code>), and the <code>val</code> should be an array of two numbers, indicating the start and end points of the time span (e.g., <code>[0 86400]</code>).

It should be noted that none of these attributes are mandatory to get a working StateSpaceModel. In its simplest form, a StateSpaceModel can have only a time span <code>t</code> and a single state <code>x.state1</code>, with all the other attributes left empty.

The StateSpaceModel class is a handle class.

### 3.1 Expected format of a StateSpaceModel object

The table below summarizes the format a StateSpaceModel object should have before any simulations have taken place. Later, the following sections, an example of a recommended way to set up such and object is given. 

In the following, a **mathematical expression** is a mathematical command in MATLAB syntax. The variables in the expression may be scalar constants (but this is not recommended, use parameters instead), or attributes of the StateSpaceModel itself, e.g. `x.state1`, `p.param1`, etc.

| Attribute | Meaning	     	  | variable type (example)	|`def` (example)		| `val` (example) 		|
| --------- | ------------------- | --------------- | -----		| ---------- |
| `t` 		| Time span 		  | DynamicElement	(`t = DyamicElement()`)| A string describing the start time of simulation (`t.def = '01-01-2001 00:00:00'`) | 2x1 array, with the start and end times (`t.val = [0 48]`) |
| `p` 		| Parameters, constant through time | Struct containing DynamicElements (`p.param1 = DyamicElement()`) | A string containing the parameter variable name, starting with `'p.'` (`p.param1.def  = 'p.param1'`) | Real scalar of the parameter value (`p.param1.val = 1`) |
| `x` 		| States, variables defined by ODEs | Struct containing DynamicElements | A string with a mathematical expression describing the ODE (`x.state1.def = '15+p.param1^2+x.state1'`) | Initial value for the state (`x.state1.val = 1`) | 
| `a` 		| Auxiliary states, any expression which helps describe the model | Struct containing DynamicElements | A string with a mathematical expression (`a.aux1.def = '15+p.param1^2'`) | Empty (`[]`) |
| `d` 		| Uncontrolled inputs (depend only on time) | Struct containing DynamicElements | A string containing the input variable name, starting with `'d.'` (`d.input1.def = 'd.input1'`) | Two-column matrix, where the first column is time and the second is the input values (`d.input1.val = [(0:47)' cos((0:47)')]`) |  
| `u` 		| Controlled inputs | Struct containing DynamicElements | **Case 1: predefined controls.** A string containing the control variable name, starting with `'u.'` (`u.control1.def = 'u.control1'`) | A two-column matrix, where the first column is time and the second is the control values (`u.control1.val = [(0:47)' zeros(size((0:47)'))]`)<br>  |
|   |   |    | **Case 2: rule-based controls.** A string with a mathematical expression defining the rule (`u.control1 = '(x.state1 < 0)*1+(x.state1 >= 0)*0''`). The functions `ifElse` and `proportionalControl` may be helpful here. | Empty, or an initial value
|   |   |     | **Case 3: optimal control using Tomlab.** A string containing the control variable name, starting with `'u.'` (`u.control1.def = 'u.control1'`) | Empty, or single value to provide a guess for the initial value, or two-column matrix to provide an initial guess for the trajectory<br>  |
| `c` 		| Constraints for an optimal control problem (optional) | Struct containing strings, each representing a single constraint, in Tomlab syntax (`c.state1 = 'final(x.state1) > 0'`, `c.ctrl1 = 'icollocate(u.ctrl1) >= 0'`)  |  | |
| `g` 		| Goal for an optimal control problem (optional) | String defining the goal, using Tomlab syntax (`g = 'final(x.state1)-integrate(u.ctrl1)'`)


### 3.2 Scheme for setting up a StateSpaceModel

By following the format in the table above, it is possible to define a StateSpaceModel by directly typing in `def`s and `val`s. However this is not advised, especially for complex models, because it will result in typing in very complex strings to set the `def`s, which can easily result in errors. Some of the nice features of MATLAB (highlighting brackets or variable names) do not work when typing strings. Thus the following scheme is recommended:
  
 1. Create an instance of the class <code>StateSpaceModel</code>.
 2. Define the time span <code>t</code>
 3. Define the parameters <code>p</code> as DynamicElements.
 4. Define the inputs <code>d</code> as DynamicElements.
 5. Define the states <code>x</code> and controls <code>u</code>. At this point the <code>def</code>s of these elements will be set as the variable name. 
 6. Define the auxiliary states <code>a</code> by using mathematical expressions.
 7. Set the ODEs by redefining the <code>def</code>s of the states <code>x</code>.
 8. Set the controls by either giving them values, or by redefining the <code>def</code>s of the controls <code>u</code>.
 9. Set the values of the inputs <code>d</code>.
 10. Set initial values for the states, and possibly for the controls. 
 11. In the case of an optimization problem, define the constraints <code>c</code> and the goal function <code>g</code>.
 
 In the following sections, we will go through this scheme, while providing examples based on Van Straten et al, *Optimal Control of Greenhouse Cultivation* (2010), Chapter 2 [1]. See file <code>example.m</code>
 
 #### 3.2.1 Create an instance of the class <code>StateSpaceModel
 
 The constructor for <code>StateSpaceModel</code> requires no arguments. It will create an empty instance of <code>StateSpaceModel</code>:
 
 > <code>>> m = StateSpaceModel()<br><br>
> m = <br>
> > <code> StateSpaceModel with properties:

> <code>x: []<br>
> a: []<br>
> d: []<br>
> p: []<br>
> u: []<br>
> c: []<br>
> g: []<br>
> t: []<br>

#### 3.2.2 Define the time span <code>t</code>
The <code>def</code> of the timespan is used to keep track of when the simulation occurs, for instance, the following can be used:

> <code>>> m.t = DynamicElement('01/01/2001 00:00:00');</code>

The <code>val</code> of the timespan should indicate a start and end point of the time span. In this case, we use hours as units, so to simulate two days, use
><code>>> m.t.val = [0 48];


#### 3.2.3 Define the parameters <code>p</code> as DynamicElements

Parameters should be the first building blocks of the model as they do not depend on other model components. <b>The definition of the parameter should be the same as the variable name.</b> Using the function `addParam` ensures that this will be the case. In this example, we have five parameters:

> <code> >> addParam(m, 'lue', 7.5e-8);<br>
> <code> >> addParam(m, 'heatLoss', 1);<br>
> <code> >> addParam(m, 'heatEff', 0.1);<br>
> <code> >> addParam(m, 'gasPrice', 4.55e-4);<br>
> <code> >> addParam(m, 'lettucePrice', 136.4);<br>

These are equivalent to the parameters c<sub>1</sub>, c<sub>2</sub>, c<sub>3</sub>, c<sub>4</sub>, c<sub>5</sub> in [1], but we use good programming practices by assigning **meaningful variable names**. This is a supported, and very useful, feature in the StateSpaceModel framework.

#### 3.2.4 Define the inputs <code>d</code> as DynamicElements

As with the parameters, the definitions should be equal to the variable names:

> <code> >> m.d.rad = DynamicElement('d.rad');<br>
> <code> >> m.d.tempOut = DynamicElement('d.tempOut');<br><br>

#### 3.2.5 Define the states <code>x</code> and controls <code>u</code>

Continuing the example of [1], we have two states and a single control. At this time, the states and controls have their <code>def</code>s equivalent to their variable names, and empty <code>val</code>s:

> <code> >> m.x.dryWeight = DynamicElement('x.dryWeight');<br>
> <code> >> m.x.tempIn = DynamicElement('x.tempIn');<br>
> <code> >> m.u.heating = DynamicElement('u.heating');<br><br>

#### 3.2.6 Define the auxiliary states <code>a</code>

The example in [1] does not use aux states, but they are often used, and could be helpful in order to give meaningful names to important processes. We can define the following:

> <code> >> % Photosynthesis [kg m^{-2} h^{-1}], equation 2.1 [1] <br>
> <code> >> m.a.phot = m.p.lue.\*m.d.rad.\*m.x.tempIn.\*m.x.dryWeight; <br><br>


> <code> >> % Heat loss to the outside [degC h^{-1}], equation 2.2 [1] <br>
> <code> >> m.a.heatOut = m.p.heatLoss.\*(m.d.tempOut - m.x.tempIn); <br><br>

> <code> >> % Heat gain from the pipes [degC h^{-1}], equation 2.2 [1] <br>
> <code> >> m.a.heatIn = m.p.heatEff.\*m.u.heating; <br><br>

What we did here was basically use the arithmetic features of DyamicElements (section 2.1.2) to define new dynamic elements:

 > <code>>> m.a.phot<br><br>
> ans = <br>
> > <code> DynamicElement with properties:
> > > <code> def: '(((p.lue).\*(d.rad)).\*(x.tempIn)).\*(x.dryWeight)'<br>
> > > <code> val: []<br>


<br>

> <code>>> m.a.heatOut<br><br>
> ans = <br>
> > <code> DynamicElement with properties:
> > > <code> def: '(p.heatLoss).\*((d.tempOut)-(x.tempIn))'<br>
> > > <code> val: []<br>

> <code>>> m.a.heatIn<br><br>
> ans = <br>
> > <code> DynamicElement with properties:
> > > <code> def: '(p.heatEff).\*(u.heating)'<br>
> > > <code> val: []<br>


#### 3.2.7 Set the ODEs by redefining the <code>def</code>
Once the auxiliary states are defined, they can be used to set the ODEs. These will be set as the <code>def</code>s of the states <code>x</code>:

> <code> >> % Photosynthesis [kg m^{-2} h^{-1}], equation 2.1 [1] <br>
> <code> >> m.x.dryWeight = DynamicElement(m.a.phot); <br><br>



> <code> >> % Heat gain in the greenhouse [degC h^{-1}], equation 2.2 [1] <br>
> <code> >> m.x.tempIn = m.a.heatOut + m.a.heatIn; <br><br>

Which makes the <code>def</code>s of the states the ODEs we want:

 > <code>>> m.x.dryWeight<br><br>
> ans = <br>
> > <code> DynamicElement with properties:
> > > <code> def: '1*((((p.lue).\*(d.rad)).\*(x.tempIn)).\*(x.dryWeight))'<br>
> > > <code> val: []<br>

> <code>>> m.x.tempIn<br><br>
> ans = <br>
> > <code> DynamicElement with properties:
> > > <code> def: '((p.heatLoss).\*((d.tempOut)-(x.tempIn)))+((p.heatEff).\*(u.heating))'<br>
> > > <code> val: []<br>


**A word of catution:** Notice that above, we used **`DynamicElement(m.a.phot)`**. This is because simply stating `m.x.dryWeight = m.a.phot` would cause `x.dryWeight` to be equivalent to `a.phot`, and any change to the one would influence the other (recall handle classes). Another solution would be to set `m.x.dryWeight.def = m.a.phot.def`.

#### 3.2.8 Set the controls

For this first example, we just assign values to the controls, in this case, all 0's, i.e., no heating input:

> <code> >> time = (0:48)';<br>
> <code> >> m.u.heating.val = [time zeros(size(time))];


#### 3.2.9 Set the values of the inputs <code>d</code>

We define the <code>val</code>s as in [1]:

> <code> >> m.d.rad.val = [time max(0, 800\*sin(4\*pi\*time/48-0.65\*pi))];<br>
> <code> >> m.d.tempOut.val = [time 15+10\*sin(4\*pi\*time/48-0.65\*pi)];<br>

#### 3.2.10 Set initial values for the states
This is necessary for solving ODEs. The initial values are set by giving a scalar (single number) to the <code>val</code> field.

> <code> >> m.x.dryWeight.val = 1; % kg m^{-2} <br> 
> <code> >> m.x.tempIn.val = 10; % degC <br><br>


## 4. Using the StateSpaceModel objects

### 4.1 Plotting

At this point we can already do some things with the model we created. Aside from looking at the various definitions, as above, try the following:

> <code> >> figure; plot(m.d.rad);<br>
> <code> >> figure; plot(m.d.tempOut);<br>
> <code> >> figure; plot(m.u.heating);<br>
> <code> >> plot(m);<br>

### 4.2 Simulating (with predefined controls)

<code>runSim</code> is a very general function, which looks at the current settings of your StateSpaceModel and simulates accordingly. Try:

> <code> >> runSim(m,1);<br>
> <code> >> plot(m);<br>

You can type <code>help runSim</code> into the MATLAB console to learn more. For instance, you can see how changing the simulation resolution influences the result.

### 4.3 Setting up control rules

In the example until now, the controls are predefined, and in fact behave exactly like inputs. However, controls can be a lot more interesting. It is possible to set rules for each control, and have `runSim` simulate the controls based on these rules. Currently, all controls must be rule based or predefined. In the future, it will be possible to have some controls predefined and some rule based.

When `runSim` is called, it checks the `def` of the first control variable. If the `def` is the same as the variable name, predefined controls are expected. In any other case, the controls can be defined, and are simulated, similar to auxiliary states. You may also set an initial value for a control by setting `val` as a scalar value.

Some helpful functions for setting control rules are `ifElse` and `proportionalControl`. These functions' help sections, as well as the included example, shows how to use these functions. In essence, `ifElse` applies bang-bang control: either fully off or fully on. In contrast, `proportionalControl` gives a smoother control, although some trial and error may be required to get good values for the setpoint and p-band.

See the example file to see how this works in practice.

### 4.4 Optimal control using Tomlab

A third option is to use Tomlab to solve an optimal control problem. For this, you need to have Tomlab installed and accessible in your MATLAB path. Tomlab is a fairly complex framework, and some basic understanding of it is recommended before use. Nevertheless, the StateSpaceModel class helps avoid some of the unpleasant things in Tomlab. For instance, in automatically converts the model as we have it so far to syntax that Tomlab understands.

Several things are still needed to define an optimal control problem, namely constraints and a goal function. For this we use the syntax of Tomlab. See the example file to see how this works, and look into the Tomlab and PROPT guides for more information.

## 5. Time-dependent variables

In principle, the system described up to now has no "memory". There is no way to directly describe a variable that depends on things that happened in the past, e.g., "state1 as it was 5 minutes ago" or "state1 in timepoint 10 of the simulation". In general there are ways to define new ODEs that collect and store information from the past, but it could be useful to have a way to access these things directly. A new functionality, **implemented so far only in the solver `runSim.m`**, can help do this. The file `exampleTimeDep.m` shows how this can be done.

### 5.1 The reserved definition 't'

The phrase `'t'` can be used for including the time influence in an auxiliary state:

> <code> >> m.a.runTime = DynamicElement('t');<br>

Will create an auxiliary state named `runTime` which simply follows the running time since the beginning of the simulation. Note that this is different from the DynamicElement `m.t` which is used only to store the timephase of the simulation (try plotting `m.t` vs plotting `m.a.runTime`). The reserved phrase `t` can be used to create any kind of variables that depend on time, you may try:

> <code> >> m.a.doubleTime = DynamicElement('2*t');<br>
> <code> >> m.a.pastTime = DynamicElement('t-10');<br>
> <code> >> m.a.timePhot = DynamicElement('t.*a.phot');<br>

### 5.2 Time dependent auxiliary states

Once an auxiliary state describing the running time has been set up, other operations can be applied to it. For instance

> <code> >> m.a.timeOfDay = mod(m.a.runTime,24);<br>

will give the time of day (assuming the simulation began at midnight). This can be used to set for instance: 

> <code> >> m.a.isMidnight = ifElse('a.timeOfDay==0',1,0);<br>

### 5.3 `atTime`

The function `atTime` (currently implemented only in runSim) enables to look into the past. The syntax for using `atTime` is as follows:

> <code> >> newDynamicElement = DynamicElement('atTime(''<state,auxiliary state, or control>'', < time dependent value using t as a variable >');

The examples in `exampleTimeDep.m` file show how this may be done. In all these cases, the `<state,auxiliary state, or control>` is `x.dryWeight`. The `<time dependent value>` may be a number (`5`), a fixed time before the current time (`t-1`, `t-10`), or a variable time before the current time (`t-a.timeOfDay`). See `exampleTimeDep.m` to see how this works in practice. Notice that variable that "look into the past" get `NaN` values if the past has not yet been simulated. For instance `m.a.dwAt5` is not defined for timepoints t<5. Similarly `m.a.dw10HoursAgo` is not defined for time t<10.